/**
 * lip-sync-data.mjs — Phoneme SVG path data for avatar mouth animation
 *
 * Each phoneme maps to 6 mouth part paths (front_background, front_tongue,
 * front_lower_teeth, front_lower_mouth, front_upper_teeth, front_upper_mouth).
 *
 * Phonemes: start (rest), ow1, ow2, t, aa, m, ax, n
 * Extracted from the original SMIL animation SVG.
 */

export const MOUTH_PARTS = [
  'front_background',
  'front_tongue',
  'front_lower_teeth',
  'front_lower_mouth',
  'front_upper_teeth',
  'front_upper_mouth'
];

/**
 * Phoneme path data. Keys are phoneme names, values are objects
 * mapping mouth part IDs to SVG path `d` attribute strings.
 */
export const PHONEMES = {
  start: {
    front_background: 'M 601,479.40 C 595.09,479.40 580.90,468.51 569.16,466.65 C 557.41,464.80 538.97,467.96 530.92,469.81 C 514.05,473.66 486.02,486.06 461,496.18 C 435.98,506.30 364.90,534.20 345.66,542.90 C 332.83,548.70 311.60,553.44 281.96,557.11 C 317.28,572.30 341.87,583.83 355.70,591.69 C 376.46,603.48 438.98,632.33 461,641.20 C 483.02,650.08 519.05,659.58 530.92,659.58 C 542.79,659.58 584.04,659.58 601,659.58 C 617.96,659.58 659.95,659.58 676.13,659.58 C 692.31,659.58 752.36,635.46 766.53,629.16 C 780.70,622.85 826.61,600.31 836.14,595.72 C 845.68,591.13 916.79,558.48 916.79,557.11 C 916.79,555.74 856.26,541.63 850,538.00 C 843.74,534.36 799.03,515.67 779,506.59 C 758.97,497.50 730.73,487.36 684.80,473.22 C 667.51,467.90 642.70,463.61 631.11,466.65 C 615.99,470.62 607.40,479.40 601,479.40 Z',
    front_tongue: 'M 397,607.79 C 397,602.41 424.90,594.99 452.15,590.32 C 479.40,585.65 510.80,582 548.05,582 C 569.21,582 587.07,585.64 603.73,585.64 C 616.40,585.64 633.13,582 652.61,582 C 697.73,582 730.09,586.85 749.80,590.32 C 769.50,593.79 810.46,602.77 810.46,612.87 C 810.46,622.97 770.73,639.11 749.80,646.54 C 728.86,653.98 686.24,655.78 652.61,655.78 C 636.90,655.78 620.41,655.78 603.86,655.78 C 584.99,655.78 566.04,655.78 548.05,655.78 C 514.26,655.78 474.66,654.40 452.15,646.54 C 429.64,638.69 397,613.17 397,607.79 Z',
    front_lower_teeth: 'M 292.08,557.57 C 292.08,554.97 343.23,565.80 353.61,568.50 C 363.99,571.20 418.99,570.61 420.66,573.00 C 422.33,575.39 418.39,593.49 419.66,593.67 C 420.94,593.84 424.79,575.66 426.50,574.83 C 428.21,574 458.77,578.67 461.24,581.49 C 463.71,584.32 463.71,606.19 464.42,606.19 C 465.13,606.19 465.14,584.66 468.28,582.06 C 471.42,579.45 500.16,582.49 501.84,585.32 C 503.53,588.14 503.43,614.25 504.60,614.29 C 505.64,614.32 505.48,587.24 508.09,585.48 C 510.70,583.72 543.09,584.34 545.62,586.95 C 548.16,589.57 547.31,617.83 548.39,617.83 C 549.46,617.83 548.83,589.51 551.11,587.48 C 553.40,585.45 596.25,585.32 597.79,587.95 C 599.34,590.59 599.03,623.83 600,623.83 C 600.95,623.83 601.52,590.19 603.67,587.95 C 605.83,585.72 646.61,585.83 649.00,588.48 C 651.39,591.13 651.50,617.83 652.52,617.83 C 653.54,617.83 652.20,589.62 654.93,587.48 C 657.66,585.34 688.58,582.49 691.75,584.72 C 694.93,586.95 696.60,612.00 697.93,612.00 C 699.26,612.00 695.93,586.95 697.88,584.34 C 699.83,581.73 729.58,579.11 732.55,581.75 C 735.52,584.39 735.29,607.46 736.29,607.43 C 737.21,607.40 736.50,584.08 738.29,581.58 C 740.09,579.08 774.69,578.93 775.13,579.36 C 775.57,579.78 778.47,597.66 779.69,597.66 C 780.56,597.66 780.03,579.25 781.00,578.36 C 781.97,577.46 828.04,577.23 837.75,574.83 C 844.22,573.22 865.03,568.95 900.17,562.00 C 873.82,579.30 855.52,591.19 845.28,597.66 C 829.93,607.37 815.18,617.83 790.05,627.63 C 764.92,637.42 745.30,650.22 721.04,655.78 C 696.78,661.34 636.56,664.25 609.53,664.25 C 582.49,664.25 522.95,664.47 501.84,658.87 C 480.74,653.26 438.92,639.02 414.43,627.63 C 389.95,616.23 372.33,610.72 362.22,602.72 C 352.12,594.72 292.08,560.16 292.08,557.57 Z',
    front_lower_mouth: 'M 281,557 C 281,556.57 284.33,557 285,557 C 285.67,557 305.45,557 309,557 C 313.99,557 294.55,557 350.05,557 C 405.55,557 431.06,557 460.74,557 C 490.42,557 563.36,559.41 600.70,559.41 C 638.04,559.41 708.25,555.63 741,555.63 C 773.75,555.63 818.96,557.10 850,557.11 C 881.04,557.12 891.49,557 893,557 C 894.24,557 910.09,556.90 915,557 C 916.09,557.02 918.63,556.87 918.72,557 C 918.81,557.13 898.12,571.27 893,574.31 C 887.88,577.36 866.09,590.38 850,600 C 833.91,609.62 797.75,630.72 777.34,639.29 C 756.94,647.86 710.79,666.22 675.68,671.44 C 640.57,676.66 621.20,671.44 601,671.44 C 580.80,671.44 560.11,676.66 529.52,671.44 C 498.93,666.22 454.32,653.32 427.23,641.30 C 400.15,629.29 352.63,603.14 346.38,600 C 340.14,596.86 322.48,585.58 309,577.02 C 295.52,568.45 281,557.43 281,557 Z',
    front_upper_teeth: 'M 416.27,517 C 417.33,517 415.49,532.00 417.21,534.00 C 418.92,536.00 464.57,545.50 467.17,544.15 C 469.78,542.79 470.04,525.94 471.16,526.01 C 471.92,526.06 472.03,541.63 474.50,544.15 C 476.97,546.66 526.73,549.94 529.20,547.97 C 531.66,546.00 532.52,527.01 533.15,527.01 C 533.79,527.01 533.99,546.68 536.35,548.97 C 538.71,551.26 592.03,553.15 595.62,551.26 C 599.21,549.37 598.86,525.01 600,525.01 C 601.14,525.01 601.03,549.23 604.22,551.26 C 607.41,553.29 662.18,552.29 663.97,549.97 C 665.76,547.65 665.55,525.08 666.73,525 C 667.91,524.92 667.92,545.82 670.73,547.97 C 673.54,550.12 722.12,547.61 725.39,545.15 C 728.66,542.67 726.82,525 728.12,525 C 729.42,525 729.77,541.38 732.14,544.15 C 734.51,546.91 783.12,538.14 784.30,535.86 C 785.47,533.59 783.14,517.01 784.30,517.01 C 785.45,517.01 787.75,532.86 789.34,534.81 C 790.93,536.75 832.19,535.05 839.58,535.86 C 846.96,536.68 885.16,547.81 881.92,545.86 C 878.68,543.91 852.49,534.31 832.49,524.38 C 812.49,514.44 784.30,504.95 753.48,491.73 C 722.67,478.51 688.69,467.04 679.20,465.08 C 669.71,463.11 639.78,455.09 624.99,465.08 C 610.20,475.06 603.77,476.69 600.75,476.69 C 598.01,476.69 587.89,469.97 575.74,465.08 C 563.59,460.18 529.27,462.39 516.71,465.08 C 504.14,467.76 428.92,501.01 417.21,504.95 C 405.49,508.90 349.75,533.60 340.53,538.31 C 334.38,541.46 324.16,545.57 309.87,550.65 C 340.12,542.03 357.42,537.11 361.79,535.86 C 368.35,534.00 407.48,534.86 410.28,532.65 C 413.07,530.43 415.22,517 416.27,517 Z',
    front_upper_mouth: 'M 309,544.07 C 300.82,547.75 281,556.75 281,557 C 281,557.25 284.16,557 285,557 C 285.85,557 334.49,557 350.68,557 C 366.87,557 408.54,555.63 421.82,555.63 C 435.09,555.63 473.42,554.47 497.32,554.47 C 521.22,554.47 575.86,557 600.24,557 C 624.62,557 682.83,553.55 709.74,553.55 C 736.64,553.55 768.05,555.63 779,555.63 C 789.95,555.63 840.67,557 850,557 C 856.89,557 914.03,557 915,557 C 915.97,557 918.72,557.16 918.72,557 C 918.72,556.37 898.73,546.79 893,544.07 C 887.27,541.36 763.02,489.72 742.08,480.68 C 721.15,471.63 704.68,466.11 682.27,458.90 C 659.87,451.68 640.94,450.79 627.43,453.73 C 613.93,456.67 604.30,468.11 601,468.11 C 597.70,468.11 588.71,458.90 573.48,454.33 C 558.26,449.77 530.47,455.96 520.09,458.90 C 509.71,461.83 496.57,466.86 467.52,478.58 C 438.48,490.30 317.18,540.40 309,544.07 Z'
  },

  ow1: {
    front_background: 'M 599.86,476.65 C 593.95,476.65 581.97,461.00 570.23,459.15 C 558.48,457.29 536.85,456.89 526.78,459.15 C 511.73,462.52 493.83,462.35 482.57,471.20 C 471.30,480.04 441.31,499.78 433.19,507.91 C 427.78,513.33 403.36,530.16 359.95,558.42 C 393.31,585.53 414.07,602.39 422.22,609.01 C 434.44,618.93 463.33,638.99 473.83,645.73 C 484.33,652.48 503.85,663.28 518.55,667.12 C 533.25,670.95 585.77,683.39 602.73,683.39 C 619.70,683.39 661.77,677.82 679.06,671.13 C 696.34,664.45 720.75,653.19 734.56,645.73 C 748.36,638.28 778.77,614.73 783.54,609.01 C 788.31,603.29 839.93,559.79 839.93,558.42 C 839.93,557.05 816.74,532.71 796.17,518.39 C 775.61,504.07 754.34,500.37 738.77,492.75 C 723.20,485.13 691.67,468.40 679.06,462.15 C 664.48,454.93 637.23,456.11 625.65,459.15 C 610.53,463.12 606.26,476.65 599.86,476.65 Z',
    front_tongue: 'M 460.89,607.91 C 460.89,593.16 479.54,588.37 502.15,580.76 C 524.75,573.14 540.54,571.28 555.44,570.08 C 572.86,568.68 583.69,566.99 600.35,566.99 C 613.02,566.99 642.75,567.36 660.22,570.76 C 677.70,574.16 693.97,577.38 703.79,580.76 C 713.61,584.13 741.07,595.85 741.07,607.91 C 741.07,619.97 712.33,645.25 698.06,653.08 C 683.78,660.91 664.16,663.96 651.38,663.96 C 638.60,663.96 615.64,667.32 605.22,667.32 C 594.79,667.32 580.56,663.96 568.48,663.96 C 556.40,663.96 545.25,661.72 521.11,656.70 C 496.97,651.68 460.89,622.66 460.89,607.91 Z',
    front_lower_teeth: 'M 351.91,561.47 C 351.91,558.87 388.34,592.03 402.58,602.09 C 416.82,612.15 468.99,621.61 470.66,624.00 C 472.33,626.39 468.39,644.49 469.66,644.67 C 470.94,644.84 474.79,626.66 476.50,625.83 C 478.21,625.00 498.77,629.67 501.24,632.49 C 503.71,635.32 503.71,657.19 504.42,657.19 C 505.13,657.19 505.14,635.66 508.28,633.06 C 511.42,630.45 530.16,633.49 531.84,636.32 C 533.53,639.14 533.43,665.25 534.60,665.29 C 535.64,665.32 535.47,638.24 538.09,636.48 C 540.71,634.72 563.09,635.34 565.62,637.96 C 568.16,640.57 567.31,668.84 568.39,668.84 C 569.46,668.84 568.83,640.51 571.11,638.48 C 573.40,636.45 596.25,636.32 597.79,638.96 C 599.34,641.59 599.03,674.84 600,674.84 C 600.95,674.84 601.52,641.19 603.67,638.96 C 605.83,636.72 626.61,636.83 629.00,639.48 C 631.39,642.13 631.50,668.84 632.52,668.84 C 633.54,668.84 632.20,640.62 634.93,638.48 C 637.66,636.34 658.58,633.49 661.75,635.72 C 664.93,637.96 666.60,663 667.93,663 C 669.26,663 665.93,637.96 667.88,635.34 C 669.83,632.73 689.58,630.11 692.55,632.75 C 695.52,635.39 695.29,658.46 696.29,658.43 C 697.21,658.40 696.50,635.08 698.29,632.58 C 700.09,630.08 724.69,629.93 725.13,630.36 C 725.57,630.78 728.47,648.66 729.69,648.66 C 730.56,648.66 730.03,630.26 731.00,629.36 C 731.97,628.46 776.98,624.86 784.32,617.16 C 789.21,612.02 811.08,593.46 849.93,561.47 C 818.96,597.11 800.20,617.96 793.66,624.00 C 783.85,633.06 763.00,651.17 746.34,663 C 729.69,674.84 721.48,680.30 692.55,692.65 C 663.62,705.01 630.71,707.00 603.67,707.00 C 576.64,707.00 556.59,705.94 531.84,698.28 C 507.10,690.61 485.34,677.34 466.01,663 C 446.69,648.66 424.00,630.65 408.20,617.16 C 392.40,603.66 351.91,564.06 351.91,561.47 Z',
    front_lower_mouth: 'M 341,557 C 341,556.57 375.25,554.58 376.41,555.12 C 377.57,555.65 388.61,558.15 399.45,563.16 C 410.30,568.17 423.15,576.28 437.29,583.12 C 451.43,589.97 511.81,621.11 542.59,627.02 C 573.36,632.93 585.16,634.42 600.35,634.42 C 615.54,634.42 640.21,635.09 670.54,627.02 C 700.88,618.96 742.86,594.28 759.41,583.12 C 775.95,571.96 780.54,567.90 790.51,563.16 C 800.47,558.42 827.29,555.02 832.20,555.12 C 833.29,555.14 858.63,556.94 858.72,557.07 C 858.81,557.20 847.28,569.48 828.12,590.97 C 808.96,612.45 791.68,629.01 775.38,643.32 C 759.08,657.63 747.84,666.89 734.22,675.53 C 720.60,684.18 687.12,698.92 670.54,702.45 C 653.97,705.97 634.48,710.68 601.07,710.68 C 567.65,710.68 546.05,707.13 529.31,702.45 C 512.56,697.76 482.40,683.63 468,675.53 C 453.60,667.44 438.74,655.03 425,643.32 C 411.26,631.61 391.64,611.53 373,590.97 C 354.36,570.40 341,557.43 341,557 Z',
    front_upper_teeth: 'M 480.28,488.96 C 481.11,488.96 482.88,503.56 483.75,503.56 C 484.63,503.56 514.57,503.50 517.17,502.14 C 519.78,500.79 520.04,483.94 521.16,484.01 C 521.92,484.06 522.03,499.63 524.50,502.14 C 526.97,504.66 545.73,507.94 548.20,505.97 C 550.66,504.00 551.52,485.01 552.15,485.01 C 552.79,485.01 552.99,504.68 555.35,506.97 C 557.71,509.26 592.03,511.15 595.62,509.26 C 599.21,507.37 598.86,483.01 600,483.01 C 601.14,483.01 601.02,507.23 604.22,509.26 C 607.41,511.29 642.18,510.29 643.97,507.97 C 645.76,505.65 645.55,483.08 646.73,483 C 647.91,482.92 647.92,503.82 650.73,505.97 C 653.54,508.12 682.12,505.61 685.39,503.14 C 688.66,500.67 686.78,483.01 688.08,483.01 C 689.38,483.01 689.77,499.38 692.14,502.14 C 694.51,504.91 729.12,506.14 730.30,503.86 C 731.47,501.59 730.29,491.67 731.45,491.67 C 732.61,491.67 733.75,500.86 735.34,502.81 C 736.93,504.75 779.63,521.54 789.79,524.49 C 799.95,527.44 833.40,551.17 833.40,550.05 C 833.40,548.92 811.17,533.68 789.79,516.77 C 768.41,499.86 737.65,481.22 715.28,469.65 C 692.92,458.08 682.10,448.74 672.61,446.77 C 663.12,444.80 640.03,441.87 627.95,446.77 C 615.87,451.67 603.52,458.51 600.50,458.51 C 597.76,458.51 582.20,450.26 568.84,446.77 C 555.49,443.29 540.75,444.46 524.50,449.58 C 508.25,454.70 490.18,461.19 468.13,477.99 C 446.08,494.80 428.98,504.58 408.70,520.39 C 399.67,527.43 385.27,537.32 365.52,550.05 C 394.05,537.15 409.81,530.31 412.81,529.52 C 417.30,528.34 478.77,504.72 480.28,504.00 C 481.79,503.28 479.44,488.96 480.28,488.96 Z',
    front_upper_mouth: 'M 371.59,534.28 C 351.83,549.21 341,556.75 341,557 C 341,557.25 358.68,557.28 364.34,557.28 C 369.99,557.28 405.65,561.32 427.42,554.43 C 449.18,547.53 475.57,539.15 498.19,532.07 C 520.80,524.99 540.27,521.64 553.01,519.36 C 565.75,517.07 586.40,514.27 601.07,514.51 C 615.73,514.74 637.30,516.86 650.70,519.36 C 664.10,521.86 678.86,524.35 703.18,532.07 C 727.51,539.78 757.21,550.20 779.99,554.43 C 802.77,558.65 822.66,557.16 830.17,557.28 C 837.68,557.40 858.72,557.45 858.72,557.28 C 858.72,556.65 853.06,549.21 830.17,534.28 C 807.29,519.36 780.32,503.51 754.76,486.23 C 729.20,468.95 703.44,454.62 678.86,443.52 C 654.27,432.42 635.51,436.78 626.07,440.19 C 616.62,443.59 602.39,452.42 600.31,452.42 C 598.24,452.42 581.16,443.20 569.43,440.19 C 557.70,437.17 537.10,437.17 516.31,444.62 C 495.52,452.06 473.47,464.19 439.68,487.23 C 405.89,510.27 391.35,519.36 371.59,534.28 Z'
  },

  ow2: {
    front_background: 'M 599.08,497.72 C 593.17,497.72 583.61,475.11 571.87,473.25 C 560.12,471.39 553.14,469.38 536.09,476.51 C 522.28,482.28 512.42,488.49 497.73,497.72 C 483.04,506.95 477.90,522.01 469.78,530.14 C 464.37,535.55 453.35,544.40 436.71,556.67 C 446.43,570.61 455.30,581.72 463.32,590 C 475.34,602.42 485.23,623.63 497.73,629.86 C 510.24,636.09 525.21,641.99 540.12,647.88 C 555.03,653.76 582.12,657.12 599.08,657.12 C 616.04,657.12 647.94,652.88 657.29,647.88 C 666.65,642.87 693.14,630.31 705.39,622.42 C 717.64,614.53 739.34,588.98 744.10,583.26 C 748.88,577.54 762.01,559.10 762.01,557.73 C 762.01,556.36 746.92,535.60 740.66,531.96 C 734.40,528.33 710.54,502.82 701.69,497.72 C 692.84,492.63 681.28,486.86 668.43,481.23 C 652.70,474.36 635.24,470.21 623.66,473.25 C 608.53,477.22 605.48,497.72 599.08,497.72 Z',
    front_tongue: 'M 472.87,560.52 C 482.28,552.57 497.94,563.30 508.59,571.65 C 519.24,580 535.54,603.76 550.12,611.80 C 562.25,618.49 587.19,626.20 603.86,626.20 C 616.53,626.20 640.63,616.97 654.78,608.65 C 668.93,600.33 682.97,582.79 698.02,571.65 C 713.08,560.52 724.77,559.62 729.14,564.36 C 733.52,569.11 720.18,595.23 708.09,608.65 C 695.99,622.07 678.46,640.28 660.77,651.28 C 643.07,662.29 620.41,667.11 603.86,667.11 C 584.99,667.11 563.16,662.18 548.05,651.28 C 532.93,640.39 511.59,627.04 499.36,611.80 C 487.13,596.57 463.46,568.46 472.87,560.52 Z',
    front_lower_teeth: 'M 426.22,562.93 C 426.22,560.33 492.37,604.97 502.75,607.67 C 513.12,610.37 538.99,595.61 540.66,598.00 C 542.34,600.39 538.39,618.49 539.66,618.67 C 540.94,618.84 544.79,600.66 546.50,599.83 C 548.21,599.00 548.77,603.67 551.24,606.49 C 553.71,609.32 553.71,631.19 554.42,631.19 C 555.13,631.19 555.14,609.66 558.28,607.06 C 561.42,604.45 560.16,607.49 561.84,610.32 C 563.53,613.14 563.43,639.25 564.60,639.29 C 565.64,639.32 565.47,612.24 568.09,610.48 C 570.71,608.72 573.09,609.34 575.62,611.96 C 578.16,614.57 577.31,642.84 578.39,642.84 C 579.46,642.84 578.83,614.51 581.11,612.48 C 583.40,610.45 596.25,610.32 597.79,612.96 C 599.34,615.59 599.03,648.84 600,648.84 C 600.95,648.84 601.52,615.19 603.67,612.96 C 605.83,610.72 616.61,610.83 619.00,613.48 C 621.39,616.13 621.50,642.84 622.52,642.84 C 623.54,642.84 622.20,614.62 624.93,612.48 C 627.66,610.34 628.58,607.49 631.75,609.72 C 634.93,611.96 636.60,637 637.93,637 C 639.26,637 635.93,611.96 637.88,609.34 C 639.83,606.73 639.58,604.11 642.55,606.75 C 645.52,609.39 645.29,632.46 646.29,632.43 C 647.21,632.40 646.50,609.08 648.29,606.58 C 650.09,604.08 654.69,603.93 655.13,604.36 C 655.57,604.78 658.47,622.66 659.69,622.66 C 660.56,622.66 660.03,604.26 661.00,603.36 C 661.97,602.46 695.91,604.89 705.62,602.48 C 712.09,600.88 732.61,589.55 767.19,568.50 C 745.93,591.61 732.23,606.73 726.09,613.87 C 716.88,624.58 697.16,633.45 691.52,637 C 685.89,640.55 671.09,649.71 659.69,654.53 C 648.29,659.36 614.29,664.93 605.65,664.93 C 597.00,664.93 581.11,664.93 564.60,661.19 C 548.09,657.45 545.17,657.55 526.75,648.84 C 508.33,640.12 490.80,631.61 485.32,626.89 C 479.84,622.17 426.22,565.52 426.22,562.93 Z',
    front_lower_mouth: 'M 411,557 C 411,556.57 442.41,556.57 453.32,557 C 464.22,557.43 482.19,557.97 494.15,561.34 C 506.10,564.72 515.12,567.75 524.78,569.59 C 534.45,571.43 549.98,575.91 563.22,577.26 C 576.46,578.61 584.81,579.39 600,579.39 C 615.19,579.39 623.94,578.48 631.76,577.26 C 639.58,576.03 661.45,572.42 673.73,569.59 C 686.01,566.76 687.58,564.95 701.43,561.34 C 715.28,557.73 731.85,557.06 738,557 C 744.15,556.94 788.63,556.94 788.72,557.07 C 788.81,557.20 774.55,575.22 759.94,592.50 C 745.34,609.79 738,617.29 723.14,629.95 C 708.27,642.60 691.24,652.61 680.78,657.41 C 670.33,662.22 658.04,666.39 646.18,669.44 C 634.34,672.49 617.08,672.95 601.40,672.95 C 585.72,672.95 576.45,671.56 564.89,669.44 C 553.32,667.32 540.37,663.88 524.78,657.41 C 509.20,650.95 495.56,642.59 479.67,629.95 C 463.77,617.30 450.40,604.27 434.93,586.11 C 419.45,567.95 411,557.43 411,557 Z',
    front_upper_teeth: 'M 541.27,509 C 542.32,509 540.49,524.00 542.21,526.00 C 543.92,528.00 551.57,533.50 554.17,532.15 C 556.78,530.79 557.04,517.94 558.16,518.01 C 558.92,518.06 559.03,533.63 561.50,536.15 C 563.97,538.66 570.73,541.94 573.20,539.97 C 575.66,538.00 576.52,519.01 577.15,519.01 C 577.79,519.01 577.99,538.68 580.35,540.97 C 582.71,543.26 592.03,545.15 595.62,543.26 C 599.21,541.37 598.86,517.01 600,517.01 C 601.14,517.01 601.02,541.23 604.22,543.26 C 607.41,545.29 619.18,544.29 620.97,541.97 C 622.76,539.65 622.55,517.08 623.73,517 C 624.91,516.92 624.92,537.82 627.73,539.97 C 630.54,542.12 639.12,539.61 642.39,537.15 C 645.66,534.67 643.82,517 645.12,517 C 646.42,517 646.77,533.38 649.14,536.15 C 651.51,538.91 662.12,530.14 663.30,527.86 C 664.47,525.59 662.14,509.01 663.30,509.01 C 664.45,509.01 665.28,527.52 668.34,526.81 C 671.40,526.09 675.29,508.31 680.89,505.39 C 686.49,502.47 743.02,560.90 743.02,560.04 C 743.02,559.19 725.97,541.50 718.16,533.87 C 709.00,524.92 697.09,508.31 683.51,498.28 C 669.92,488.25 660.19,478.67 651.72,474.87 C 643.26,471.07 629.47,466.42 619.46,472.87 C 609.45,479.32 603.37,488.41 600.35,488.41 C 597.61,488.41 583.15,474.58 574.50,472.17 C 565.85,469.77 558.33,468.44 546.08,474.87 C 533.83,481.30 521.44,491.19 513.16,498.28 C 504.88,505.37 493.96,518.95 486.95,526.09 C 482.28,530.85 473.22,540.37 459.78,554.65 C 495.99,518.71 516.10,501.32 520.13,502.47 C 526.16,504.20 533.22,525.08 536.01,522.87 C 538.80,520.65 540.22,509 541.27,509 Z',
    front_upper_mouth: 'M 439.93,535.65 C 425.27,545.90 411,556.75 411,557 C 411,557.25 425.26,561.29 436.71,562.82 C 448.15,564.34 468.27,571.37 485.21,567.98 C 502.15,564.59 520.42,560.73 534.24,557.73 C 548.06,554.73 556.33,553.36 567.88,552.86 C 579.43,552.35 586.39,554.73 600.35,554.73 C 614.31,554.73 618.32,552.55 631.75,552.86 C 645.18,553.16 651.22,553.54 670.26,557.28 C 689.30,561.02 706.25,566.51 721.69,568 C 737.13,569.49 757.33,564.43 766,562.82 C 774.67,561.21 788.72,556.88 788.72,557.28 C 788.72,557.68 772.86,546.55 762.01,538.36 C 751.16,530.17 740.76,523.18 721.69,510.20 C 702.63,497.21 683.96,483.05 664.62,476.20 C 645.27,469.35 629.00,469.78 619.56,473.19 C 610.11,476.59 602.42,482.41 600.35,482.41 C 598.28,482.41 591.16,476.20 579.43,473.19 C 567.70,470.17 555.23,468.44 534.24,476.20 C 513.26,483.96 491.42,497.13 474.29,510.20 C 457.16,523.26 454.59,525.41 439.93,535.65 Z'
  },

  t: {
    front_background: 'M 602.73,496.22 C 596.82,496.22 584.43,483.23 572.68,481.38 C 560.94,479.52 538.97,482.45 530.92,484.29 C 514.05,488.15 482.24,499.01 457.21,509.13 C 432.19,519.25 369.35,534.66 345.66,543.86 C 329.87,549.98 311.96,554.40 291.96,557.11 C 315.49,569.63 333.39,582.21 345.66,594.83 C 364.07,613.77 438.11,657.68 457.21,667.73 C 476.32,677.77 520.92,693.82 530.92,698.05 C 540.92,702.28 583.04,709.39 600,709.39 C 616.96,709.39 665.47,695.22 679,693.04 C 692.53,690.86 764.04,656.97 779,646.84 C 793.96,636.71 848.36,603.83 857.10,594.83 C 865.84,585.83 906.79,558.48 906.79,557.11 C 906.79,555.74 862.03,545.76 851.54,542.73 C 841.06,539.70 799.03,529.13 779,520.04 C 758.97,510.95 732.50,501.42 686.57,487.29 C 669.28,481.97 647.46,478.34 635.88,481.38 C 620.76,485.35 609.14,496.22 602.73,496.22 Z',
    front_tongue: 'M 397.26,614.93 C 397.26,605.72 410.19,587.83 432.15,577.31 C 454.11,566.78 519.23,544.43 526.74,540.49 C 534.25,536.55 580.13,519.77 601,521.25 C 621.87,522.73 667.98,536.91 674.60,540.49 C 681.21,544.07 757.51,568.41 774.84,577.31 C 792.18,586.20 811.37,594.45 811.37,614.93 C 811.37,635.40 771.38,654.57 749.80,670.27 C 728.22,685.97 705.52,694.21 692.34,700.23 C 684.56,703.78 620.28,713.47 603.73,713.47 C 584.87,713.47 533.46,705.89 519.71,700.23 C 500.23,692.21 476.27,677.94 452.15,663.44 C 428.03,648.94 397.26,624.14 397.26,614.93 Z',
    front_lower_teeth: 'M 310.77,572.01 C 315.42,574.52 344.35,586.40 360.94,589.11 C 377.53,591.81 415.99,605.61 417.66,608.00 C 419.33,610.39 418.39,624.49 419.66,624.67 C 420.94,624.84 421.79,610.66 423.50,609.83 C 425.21,609.00 458.77,609.67 461.24,612.49 C 463.71,615.32 463.71,637.19 464.42,637.19 C 465.13,637.19 465.14,615.66 468.28,613.06 C 471.42,610.45 500.16,613.49 501.84,616.32 C 503.53,619.14 503.43,645.25 504.60,645.29 C 505.64,645.32 505.48,618.24 508.09,616.48 C 510.70,614.72 543.09,615.34 545.62,617.96 C 548.16,620.57 547.31,648.84 548.39,648.84 C 549.46,648.84 548.83,620.51 551.11,618.48 C 553.40,616.45 596.25,616.32 597.79,618.96 C 599.34,621.59 599.03,654.84 600,654.84 C 600.95,654.84 601.52,621.19 603.67,618.96 C 605.83,616.72 646.61,616.83 649.00,619.48 C 651.39,622.13 651.50,648.84 652.52,648.84 C 653.54,648.84 652.20,620.62 654.93,618.48 C 657.66,616.34 688.58,613.49 691.75,615.72 C 694.93,617.96 696.60,643 697.93,643 C 699.26,643 695.93,617.96 697.88,615.34 C 699.83,612.73 729.58,610.11 732.55,612.75 C 735.52,615.39 735.29,638.46 736.29,638.43 C 737.21,638.40 736.50,615.08 738.29,612.58 C 740.09,610.08 774.69,609.93 775.13,610.36 C 775.57,610.78 778.47,628.66 779.69,628.66 C 780.56,628.66 780.03,610.26 781.00,609.36 C 781.97,608.46 839.97,596.40 849.67,594.00 C 856.14,592.39 871.07,585.06 894.44,572.01 C 871.14,597.07 856.21,612.62 849.67,618.66 C 839.87,627.72 813.74,650.94 797.77,664.69 C 781.81,678.43 759.92,693.33 730.55,708.84 C 701.18,724.35 629.30,737.27 602.27,737.27 C 575.23,737.27 524.75,727.65 499.84,715.74 C 474.94,703.84 423.74,677.07 409.41,664.69 C 395.09,652.30 367.59,628.16 351.79,614.67 C 335.99,601.17 306.12,569.50 310.77,572.01 Z',
    front_lower_mouth: 'M 281,557 C 281,556.57 286.18,556.46 287.34,557 C 288.50,557.54 304.92,558.95 309.84,560.05 C 314.76,561.16 335.36,563.90 362.18,579.29 C 389.00,594.69 433.56,613.71 459.98,625.77 C 486.39,637.84 547.19,655.73 601,658.39 C 654.81,661.05 728.66,640.58 762.27,625.77 C 795.88,610.97 827.11,593.13 846.58,579.29 C 866.05,565.46 892.75,558.24 895.53,558.01 C 898.30,557.77 910.09,556.90 915,557 C 916.09,557.02 918.63,556.87 918.72,557 C 918.81,557.13 904.21,568.01 897.65,575.73 C 891.10,583.45 877.17,597.24 862.88,614.39 C 848.59,631.54 817.83,662.86 787.27,684.75 C 756.71,706.64 707.36,730.43 679,737.50 C 650.64,744.57 624.37,746.74 601,746.74 C 577.63,746.74 556.91,742.22 524,732.83 C 491.09,723.44 452.11,703.72 425,684.75 C 397.89,665.78 367.00,636.60 353.64,622.74 C 340.28,608.88 315.84,585.43 306.67,577.87 C 297.50,570.31 281,557.43 281,557 Z',
    front_upper_teeth: 'M 416.27,521 C 417.33,521 415.49,536.00 417.21,538.00 C 418.92,540.00 464.57,549.50 467.17,548.15 C 469.78,546.79 470.04,529.94 471.16,530.01 C 471.92,530.06 472.03,545.63 474.50,548.15 C 476.97,550.66 526.73,553.94 529.20,551.97 C 531.66,550.00 532.52,531.01 533.15,531.01 C 533.79,531.01 533.99,550.68 536.35,552.97 C 538.71,555.26 592.03,557.15 595.62,555.26 C 599.21,553.37 598.86,529.01 600,529.01 C 601.14,529.01 601.03,553.23 604.22,555.26 C 607.41,557.29 662.18,556.29 663.97,553.97 C 665.76,551.65 665.55,529.08 666.73,529 C 667.91,528.92 667.92,549.82 670.73,551.97 C 673.54,554.12 722.12,551.61 725.39,549.15 C 728.66,546.67 726.82,529 728.12,529 C 729.42,529 729.77,545.38 732.14,548.15 C 734.51,550.91 783.12,542.14 784.30,539.86 C 785.47,537.59 783.14,521.01 784.30,521.01 C 785.45,521.01 787.75,536.86 789.34,538.81 C 790.93,540.75 828.59,546.00 838.40,546.00 C 848.21,546.00 878.57,551.45 875.33,549.50 C 872.09,547.55 832.96,531.65 825.16,529.01 C 808.11,523.24 789.34,510.09 766.95,500.60 C 744.57,491.10 705.68,476.20 695.57,472.13 C 685.46,468.07 650.58,468.07 635.88,471.38 C 621.18,474.68 607.24,490.64 604.22,490.64 C 601.48,490.64 588.56,478.95 568.89,470.27 C 549.22,461.59 511.22,470.27 497.25,475.27 C 483.28,480.27 432.58,506.14 420.87,510.09 C 409.16,514.04 367.89,534.40 359.32,536.64 C 353.61,538.14 337.26,543.64 310.29,553.15 C 338.71,548.72 355.06,546.50 359.32,546.50 C 365.72,546.50 407.48,538.86 410.28,536.65 C 413.07,534.43 415.22,521 416.27,521 Z',
    front_upper_mouth: 'M 310.34,547.20 C 302.29,550.22 281,556.75 281,557 C 281,557.25 290.57,558.03 294.17,558.03 C 297.77,558.03 315.22,558.03 329.37,558.03 C 343.51,558.03 405.20,548.51 425.17,546.92 C 445.13,545.32 494.16,541.70 517.79,541.70 C 541.43,541.70 570.93,541.70 603.73,541.70 C 636.53,541.70 669.86,541.70 695.35,541.70 C 720.83,541.70 763.73,545.02 785.27,546.92 C 806.80,548.82 839.81,555.97 862.36,557 C 884.91,558.03 903.32,558.03 905.02,558.03 C 906.72,558.03 918.72,557.17 918.72,557 C 918.72,556.37 871.73,541.46 866,538.75 C 860.27,536.03 782.70,502.90 754.16,491.34 C 725.61,479.77 716.05,474.36 695.35,467.27 C 674.64,460.18 656.34,461.76 643.67,464.27 C 631.00,466.78 602.81,482.22 600.73,482.22 C 598.66,482.22 580.53,467.27 564.11,464.27 C 547.68,461.27 529.01,460.12 509.60,467.27 C 490.18,474.42 473.48,482.52 441.04,494.99 C 408.59,507.46 318.39,544.19 310.34,547.20 Z'
  },

  aa: {
    front_background: 'M 601,469.40 C 595.09,469.40 584.67,458.51 572.93,456.65 C 561.18,454.80 538.97,457.96 530.92,459.81 C 514.05,463.67 486.02,476.06 461,486.18 C 435.98,496.30 368.90,523.20 349.66,531.90 C 336.83,537.70 315.60,546.10 285.96,557.11 C 316.15,591.11 337.39,614.42 349.66,627.04 C 368.07,645.98 438.11,707.50 457.21,717.54 C 476.32,727.58 520.92,743.34 530.92,747.58 C 540.92,751.81 584.04,760.57 601,760.57 C 617.96,760.57 665.47,753.22 679,751.04 C 692.53,748.86 764.04,714.97 779,704.84 C 793.96,694.71 848.10,645.04 856.84,636.04 C 865.58,627.04 918.57,558.95 918.57,557.59 C 918.57,556.22 856.26,531.63 850,528.00 C 843.74,524.36 799.03,505.67 779,496.59 C 758.97,487.50 730.73,477.36 684.80,463.22 C 667.51,457.90 646.30,453.61 634.72,456.65 C 619.59,460.62 607.40,469.40 601,469.40 Z',
    front_tongue: 'M 364.41,645.82 C 340.53,616.21 424.09,588.03 452.15,580.32 C 480.21,572.61 510.80,572 548.05,572 C 569.21,572 587.07,575.64 603.73,575.64 C 616.40,575.64 633.13,572 652.61,572 C 697.73,572 725.99,573.15 749.80,580.32 C 773.61,587.49 849.33,620.81 832.00,652.74 C 814.67,684.67 770.73,712.83 749.80,720.27 C 728.86,727.70 673.57,744.21 660.39,750.23 C 652.61,753.78 620.28,750.23 603.73,750.23 C 584.87,750.23 567.92,755.89 554.17,750.23 C 534.69,742.21 474.66,721.29 452.15,713.44 C 429.64,705.59 388.29,675.44 364.41,645.82 Z',
    front_lower_teeth: 'M 297.32,572 C 297.32,569.40 354.03,643.12 364.41,645.82 C 374.79,648.52 418.99,678.61 420.66,681.00 C 422.33,683.39 418.39,701.49 419.66,701.67 C 420.94,701.84 424.79,683.66 426.50,682.83 C 428.21,682.00 458.77,686.67 461.24,689.49 C 463.71,692.32 463.71,714.19 464.42,714.19 C 465.13,714.19 465.14,692.66 468.28,690.06 C 471.42,687.45 500.16,690.49 501.84,693.32 C 503.53,696.14 503.43,722.25 504.60,722.29 C 505.64,722.32 505.48,695.24 508.09,693.48 C 510.70,691.72 543.09,692.34 545.62,694.96 C 548.16,697.57 547.31,725.84 548.39,725.84 C 549.46,725.84 548.83,697.51 551.11,695.48 C 553.40,693.45 596.25,693.32 597.79,695.96 C 599.34,698.59 599.03,731.84 600,731.84 C 600.95,731.84 601.52,698.19 603.67,695.96 C 605.83,693.72 646.61,693.83 649.00,696.48 C 651.39,699.13 651.50,725.84 652.52,725.84 C 653.54,725.84 652.20,697.62 654.93,695.48 C 657.66,693.34 688.58,690.49 691.75,692.72 C 694.93,694.96 696.60,720 697.93,720 C 699.26,720 695.93,694.96 697.88,692.34 C 699.83,689.73 729.58,687.11 732.55,689.75 C 735.52,692.39 735.29,715.46 736.29,715.43 C 737.21,715.40 736.50,692.08 738.29,689.58 C 740.09,687.08 774.69,686.93 775.13,687.36 C 775.57,687.78 778.47,705.66 779.69,705.66 C 780.56,705.66 780.03,687.25 781.00,686.36 C 781.97,685.46 820.01,659.39 829.71,656.98 C 836.18,655.38 861.11,627.05 904.50,572 C 855.92,642.20 828.35,680.32 821.82,686.36 C 812.01,695.42 795.66,708.54 779.69,722.29 C 763.73,736.03 749.96,749.20 720.58,764.71 C 691.21,780.22 632.29,793.85 605.25,793.85 C 578.21,793.85 533.00,782.81 508.09,770.90 C 483.18,758.99 445.55,744.22 431.23,731.84 C 416.91,719.45 401.78,699.85 385.98,686.36 C 370.18,672.86 297.32,574.60 297.32,572 Z',
    front_lower_mouth: 'M 281,557 C 281,556.57 288.18,556.46 289.34,557 C 290.50,557.54 305.72,573.21 311.64,578.01 C 317.56,582.80 327.27,592.35 346.20,604.87 C 365.14,617.39 434.97,665.83 461.38,677.89 C 487.79,689.96 543.40,716.67 603.04,716.67 C 662.69,716.67 705.01,692.69 738.62,677.89 C 772.24,663.09 828.47,625.11 855.11,604.87 C 881.75,584.63 885.32,581.91 889.89,578.01 C 894.47,574.10 911.09,556.90 916,557 C 917.09,557.02 921.63,556.87 921.72,557 C 921.81,557.13 902.78,588.97 898.46,595.38 C 894.13,601.79 876.82,627.18 863.22,644.97 C 849.62,662.76 805.42,712.40 780.43,733.09 C 755.44,753.78 716.89,776.19 692.01,784.90 C 667.13,793.61 626.41,800.56 603.04,800.56 C 579.67,800.56 535.32,791.83 516.17,784.90 C 497.02,777.97 458.77,760.57 423.64,733.09 C 388.51,705.60 372.63,680.70 353.64,659.00 C 334.64,637.30 318.29,613.94 306.67,596.72 C 295.05,579.50 281,557.43 281,557 Z',
    front_upper_teeth: 'M 417.28,503.96 C 418.11,503.96 419.88,518.56 420.75,518.56 C 421.63,518.56 464.57,508.50 467.17,507.14 C 469.78,505.79 470.04,488.94 471.16,489.01 C 471.92,489.06 472.03,504.63 474.50,507.14 C 476.97,509.66 526.73,512.94 529.20,510.97 C 531.66,509.00 532.52,490.01 533.15,490.01 C 533.79,490.01 533.99,509.68 536.35,511.97 C 538.71,514.26 592.03,516.15 595.62,514.26 C 599.21,512.37 598.86,488.01 600,488.01 C 601.14,488.01 601.03,512.23 604.22,514.26 C 607.41,516.29 662.18,515.29 663.97,512.97 C 665.76,510.65 665.55,488.08 666.73,488 C 667.91,487.92 667.92,508.82 670.73,510.97 C 673.54,513.12 722.12,510.61 725.39,508.14 C 728.66,505.67 726.78,488.01 728.08,488.01 C 729.38,488.01 729.77,504.38 732.14,507.14 C 734.51,509.91 779.12,511.14 780.30,508.86 C 781.47,506.59 780.29,496.67 781.45,496.67 C 782.61,496.67 783.75,505.86 785.34,507.81 C 786.93,509.75 837.96,531.31 848.12,534.26 C 858.28,537.21 911.78,555.93 911.78,554.81 C 911.78,553.68 827.06,507.59 821.99,504.67 C 810.98,498.34 777.34,481.94 754.97,470.37 C 732.61,458.80 680.22,445.16 670.73,443.19 C 661.24,441.22 651.68,438.16 632.40,440.71 C 613.11,443.26 603.02,451.78 600,451.78 C 597.26,451.78 586.03,444.19 572.67,440.71 C 559.32,437.23 541.76,440.50 529.20,443.19 C 516.63,445.88 484.43,457.65 417.28,486.00 C 397.23,494.46 371.53,512.61 352.23,520.26 C 336.37,526.55 315.24,538.06 288.83,554.81 C 324.53,542.60 343.88,536.10 346.88,535.31 C 351.37,534.13 415.77,519.72 417.28,519.00 C 418.79,518.28 416.44,503.96 417.28,503.96 Z',
    front_upper_mouth: 'M 308.77,539.57 C 300.22,544.03 281,556.75 281,557 C 281,557.25 287.64,556.77 289.40,557.29 C 291.16,557.81 332.15,545.38 354.66,540.57 C 377.18,535.76 405.09,529.19 420.05,526.25 C 435.01,523.30 475.98,518.82 499.88,518.82 C 523.78,518.82 576.62,516.91 601,516.91 C 625.38,516.91 688.87,517.02 716.06,518.82 C 743.25,520.62 770.47,524.58 782.54,526.25 C 794.61,527.91 842.77,537.93 852.33,540.57 C 861.89,543.20 912.53,556.54 914.05,557.29 C 915.56,558.03 921.72,557.16 921.72,557 C 921.72,556.37 900.61,544.97 893.48,540.57 C 886.34,536.16 765.44,469.26 738.84,456.86 C 712.24,444.45 697.74,439.20 678.58,435.20 C 659.43,431.20 641.80,429.67 629.13,432.19 C 616.46,434.70 603.16,444.64 601.09,444.64 C 599.01,444.64 584.76,435.20 570.91,432.19 C 557.06,429.17 538.41,430.28 521.29,435.20 C 504.17,440.12 498.04,442.50 465.39,456.86 C 432.74,471.21 317.32,535.11 308.77,539.57 Z'
  },

  m: {
    front_background: 'M 601,489.40 C 595.09,489.40 580.90,478.51 569.16,476.65 C 557.41,474.80 538.97,477.96 530.92,479.81 C 514.05,483.66 486.02,496.06 461,506.18 C 435.98,516.29 364.90,543.20 345.66,551.90 C 332.83,557.70 315.29,559.85 293.03,558.36 C 320.98,565.38 341.87,572.83 355.70,580.69 C 376.46,592.48 438.98,621.33 461,630.20 C 483.02,639.08 519.05,648.58 530.92,648.58 C 542.79,648.58 584.04,648.58 601,648.58 C 617.96,648.58 659.95,648.58 676.13,648.58 C 692.31,648.58 752.36,624.46 766.53,618.16 C 780.70,611.85 826.61,589.31 836.14,584.72 C 845.68,580.13 900.17,559.72 900.17,558.36 C 900.17,556.99 856.26,551.63 850,548.00 C 843.74,544.36 799.03,525.67 779,516.59 C 758.97,507.50 730.73,497.36 684.80,483.22 C 667.51,477.90 642.70,473.61 631.11,476.65 C 615.99,480.62 607.40,489.40 601,489.40 Z',
    front_tongue: 'M 397,594.79 C 397,589.41 424.90,581.99 452.15,577.32 C 479.40,572.65 510.80,569 548.05,569 C 569.21,569 587.07,572.64 603.73,572.64 C 616.40,572.64 633.13,569 652.61,569 C 697.73,569 730.09,573.85 749.80,577.32 C 769.50,580.79 810.46,589.77 810.46,599.87 C 810.46,609.97 770.73,626.11 749.80,633.54 C 728.86,640.98 686.24,642.78 652.61,642.78 C 636.90,642.78 620.41,642.78 603.86,642.78 C 584.99,642.78 566.04,642.78 548.05,642.78 C 514.26,642.78 474.66,641.40 452.15,633.54 C 429.64,625.69 397,600.17 397,594.79 Z',
    front_lower_teeth: 'M 306.50,568.50 C 306.50,565.90 350.29,568.99 362.22,568.50 C 374.16,568.01 418.99,570.61 420.66,573.00 C 422.33,575.39 418.39,593.49 419.66,593.67 C 420.94,593.84 424.79,575.66 426.50,574.83 C 428.21,574.00 458.77,578.67 461.24,581.49 C 463.71,584.32 463.71,606.19 464.42,606.19 C 465.13,606.19 465.14,584.66 468.28,582.06 C 471.42,579.45 500.16,582.49 501.84,585.32 C 503.53,588.14 503.43,614.25 504.60,614.29 C 505.64,614.32 505.48,587.24 508.09,585.48 C 510.70,583.72 543.09,584.34 545.62,586.96 C 548.16,589.57 547.31,617.84 548.39,617.84 C 549.46,617.84 548.83,589.51 551.11,587.48 C 553.40,585.45 596.25,585.32 597.79,587.96 C 599.34,590.59 599.03,623.84 600,623.84 C 600.95,623.84 601.52,590.19 603.67,587.96 C 605.83,585.72 646.61,585.83 649.00,588.48 C 651.39,591.13 651.50,617.84 652.52,617.84 C 653.54,617.84 652.20,589.62 654.93,587.48 C 657.66,585.34 688.58,582.49 691.75,584.72 C 694.93,586.96 696.60,612 697.93,612 C 699.26,612 695.93,586.96 697.88,584.34 C 699.83,581.73 729.58,579.11 732.55,581.75 C 735.52,584.39 735.29,607.46 736.29,607.43 C 737.21,607.40 736.50,584.08 738.29,581.58 C 740.09,579.08 774.69,578.93 775.13,579.36 C 775.57,579.78 778.47,597.66 779.69,597.66 C 780.56,597.66 780.03,579.25 781.00,578.36 C 781.97,577.46 828.04,577.23 837.75,574.83 C 844.22,573.22 865.03,568.95 900.17,562 C 873.17,576.64 854.55,587.20 844.31,593.67 C 828.96,603.37 815.18,611.84 790.05,621.63 C 764.92,631.42 745.30,641.22 721.04,646.78 C 696.78,652.34 631.56,656.25 604.53,656.25 C 577.49,656.25 522.95,656.47 501.84,650.87 C 480.74,645.26 437.92,633.02 413.43,621.63 C 388.95,610.23 372.33,608.72 362.22,600.72 C 352.12,592.72 306.50,571.10 306.50,568.50 Z',
    front_lower_mouth: 'M 281,560 C 281,559.57 284.33,560 285,560 C 285.67,560 305.45,557 309,557 C 313.99,557 294.55,557 350.05,557 C 405.55,557 431.06,557 460.74,557 C 490.42,557 563.36,559.41 600.70,559.41 C 638.04,559.41 708.25,555.63 741,555.63 C 773.75,555.63 818.96,557.10 850,557.11 C 881.04,557.12 891.49,557 893,557 C 894.24,557 910.09,559.90 915,560 C 916.09,560.02 918.63,559.87 918.72,560 C 918.81,560.13 898.12,571.27 893,574.31 C 887.88,577.36 866.09,587.49 850,597.11 C 833.91,606.73 797.75,620.72 777.34,629.29 C 756.94,637.86 710.79,656.22 675.68,661.44 C 640.57,666.66 621.20,661.44 601,661.44 C 580.80,661.44 560.11,666.66 529.52,661.44 C 498.93,656.22 456.06,645.52 428.97,633.51 C 401.89,621.49 352.70,600.24 346.46,597.11 C 340.21,593.97 322.48,585.58 309,577.02 C 295.52,568.45 281,560.43 281,560 Z',
    front_upper_teeth: 'M 416.27,523 C 417.33,523 415.49,538.00 417.21,540.00 C 418.92,542.00 464.57,551.50 467.17,550.15 C 469.78,548.79 470.04,531.94 471.16,532.01 C 471.92,532.06 472.03,547.63 474.50,550.15 C 476.97,552.66 526.73,555.94 529.20,553.97 C 531.66,552.00 532.52,533.01 533.15,533.01 C 533.79,533.01 533.99,552.68 536.35,554.97 C 538.71,557.26 592.03,559.15 595.62,557.26 C 599.21,555.37 598.86,531.01 600,531.01 C 601.14,531.01 601.02,555.23 604.22,557.26 C 607.41,559.29 662.18,558.29 663.97,555.97 C 665.76,553.65 665.55,531.08 666.73,531 C 667.91,530.92 667.92,551.82 670.73,553.97 C 673.54,556.12 722.12,553.61 725.39,551.15 C 728.66,548.67 726.82,531 728.12,531 C 729.42,531 729.77,547.38 732.14,550.15 C 734.51,552.91 783.12,544.14 784.30,541.86 C 785.47,539.59 783.14,523.01 784.30,523.01 C 785.45,523.01 787.75,538.86 789.34,540.81 C 790.93,542.75 832.19,541.05 839.58,541.86 C 846.96,542.68 885.16,553.81 881.92,551.86 C 878.68,549.91 852.49,540.31 832.49,530.38 C 812.49,520.44 784.30,510.95 753.48,497.73 C 722.67,484.51 688.69,473.04 679.20,471.08 C 669.71,469.11 639.78,461.09 624.99,471.08 C 610.20,481.06 603.77,482.69 600.75,482.69 C 598.01,482.69 587.89,475.97 575.74,471.08 C 563.59,466.18 529.27,468.39 516.71,471.08 C 504.14,473.76 428.92,507.01 417.21,510.95 C 405.49,514.90 349.75,539.60 340.53,544.31 C 334.38,547.46 324.16,551.57 309.87,556.65 C 340.12,548.03 357.42,543.11 361.79,541.86 C 368.35,540.00 407.48,540.86 410.28,538.65 C 413.07,536.43 415.22,523 416.27,523 Z',
    front_upper_mouth: 'M 317.76,544.07 C 309.58,547.75 281,559.75 281,560 C 281,560.25 284.16,560 285,560 C 285.85,560 334.49,558 350.68,558 C 366.87,558 408.54,556.63 421.82,556.63 C 435.09,556.63 473.42,559.47 497.32,559.47 C 521.22,559.47 576.62,568.01 601,568.01 C 625.38,568.01 682.83,558.55 709.74,558.55 C 736.64,558.55 768.05,555.63 779,555.63 C 789.95,555.63 840.67,558 850,558 C 856.89,558 914.03,560 915,560 C 915.97,560 918.72,560.16 918.72,560 C 918.72,559.37 887.84,546.79 882.11,544.07 C 876.38,541.36 763.02,499.72 742.08,490.68 C 721.15,481.63 704.68,476.11 682.27,468.90 C 659.87,461.68 640.94,460.79 627.43,463.73 C 613.93,466.67 604.30,476.11 601,476.11 C 597.70,476.11 588.71,468.90 573.48,464.33 C 558.26,459.77 530.47,465.96 520.09,468.90 C 509.71,471.83 496.57,476.86 467.52,488.58 C 438.48,500.30 325.94,540.40 317.76,544.07 Z'
  },

  ax: {
    front_background: 'M 601.83,521.09 C 595.92,521.09 569.22,503.04 557.48,501.18 C 545.73,499.33 528.75,499.34 520.70,501.18 C 503.83,505.04 478.98,512.54 457.21,517.95 C 435.45,523.36 365.42,542.69 348.86,546.73 C 337.83,549.43 319.49,553.18 293.86,558 C 318.82,568.04 339.76,578.98 356.66,590.83 C 382.01,608.60 438.11,656.80 457.21,666.84 C 476.32,676.88 520.62,689.70 530.62,693.93 C 540.62,698.17 582.90,708.05 599.86,708.05 C 616.82,708.05 665.47,702.22 679,700.04 C 692.53,697.86 764.04,663.97 779,653.84 C 793.96,643.71 848.36,601.83 857.10,592.83 C 865.84,583.83 901.06,560.59 901.06,559.22 C 901.06,557.85 862.67,549.17 851.54,546.73 C 840.41,544.30 796.65,533.37 779,528.65 C 761.35,523.92 733.45,515.32 687.52,501.18 C 670.24,495.86 646.44,502.25 634.86,505.29 C 619.73,509.26 608.23,521.09 601.83,521.09 Z',
    front_tongue: 'M 397,607.79 C 397,602.41 424.90,594.99 452.15,590.32 C 479.40,585.65 510.80,582 548.05,582 C 569.21,582 587.07,585.64 603.73,585.64 C 616.40,585.64 633.13,582 652.61,582 C 697.73,582 730.09,586.85 749.80,590.32 C 769.50,593.79 810.46,602.77 810.46,612.87 C 810.46,622.97 770.73,662.83 749.80,670.27 C 728.86,677.70 673.57,694.21 660.39,700.23 C 652.61,703.78 620.28,700.23 603.73,700.23 C 584.87,700.23 567.92,705.89 554.17,700.23 C 534.69,692.21 474.66,671.29 452.15,663.44 C 429.64,655.59 397,613.17 397,607.79 Z',
    front_lower_teeth: 'M 303.32,574.00 C 303.32,571.40 348.36,591.30 358.73,594 C 369.11,596.70 418.99,602.61 420.66,605.00 C 422.33,607.39 418.39,625.49 419.66,625.67 C 420.94,625.84 424.79,607.66 426.50,606.83 C 428.21,606.00 458.77,610.67 461.24,613.49 C 463.71,616.32 463.71,638.19 464.42,638.19 C 465.13,638.19 465.14,616.66 468.28,614.06 C 471.42,611.45 500.16,614.49 501.84,617.32 C 503.53,620.14 503.43,646.25 504.60,646.29 C 505.64,646.32 505.48,619.24 508.09,617.48 C 510.70,615.72 543.09,616.34 545.62,618.96 C 548.16,621.57 547.31,649.84 548.39,649.84 C 549.46,649.84 548.83,621.51 551.11,619.48 C 553.40,617.45 596.25,617.32 597.79,619.96 C 599.34,622.59 599.03,655.84 600,655.84 C 600.96,655.84 601.52,622.19 603.67,619.96 C 605.83,617.72 646.61,617.83 649.00,620.48 C 651.39,623.13 651.50,649.84 652.52,649.84 C 653.54,649.84 652.20,621.62 654.93,619.48 C 657.66,617.34 688.58,614.49 691.76,616.72 C 694.93,618.96 696.60,644 697.93,644 C 699.26,644 695.93,618.96 697.88,616.34 C 699.83,613.73 729.58,611.11 732.55,613.75 C 735.52,616.39 735.29,639.46 736.29,639.43 C 737.21,639.40 736.50,616.08 738.29,613.58 C 740.09,611.08 774.69,610.93 775.13,611.36 C 775.57,611.78 778.47,629.66 779.69,629.66 C 780.56,629.66 780.03,611.25 781.00,610.36 C 781.97,609.46 821.22,607.40 830.93,605.00 C 837.40,603.39 859.17,593.06 896.26,574.00 C 869.02,602.75 852.14,620.15 845.60,626.19 C 835.79,635.25 807.97,658.92 790.57,669.63 C 773.17,680.33 742.95,695.84 721.13,707.46 C 699.31,719.08 632.85,731.43 605.81,731.43 C 578.78,731.43 522.95,715.17 501.84,709.57 C 480.74,703.97 429.67,677.87 415.34,665.48 C 401.02,653.10 379.16,638.69 363.36,625.19 C 347.56,611.69 303.32,576.59 303.32,574.00 Z',
    front_lower_mouth: 'M 281,557 C 281,556.57 285.44,557 287.34,557 C 289.24,557 304.92,558.95 309.84,560.05 C 314.76,561.15 326.82,560.90 353.64,576.29 C 380.46,591.69 431.58,618.80 464.42,627.19 C 497.26,635.58 549.92,645.44 603.73,648.10 C 657.54,650.76 709.83,635.50 748.28,622.55 C 786.74,609.59 830.65,587.63 848.67,576.29 C 866.70,564.96 892.52,557.10 895.56,557 C 898.60,556.90 910.09,556.90 915,557 C 916.09,557.02 918.63,556.87 918.72,557 C 918.81,557.13 897.45,578.78 889.94,588.93 C 882.43,599.08 873.54,609.59 857.05,627.19 C 840.57,644.79 801.38,675.97 779.44,689.16 C 757.51,702.36 710.88,726.43 679,732.50 C 647.12,738.58 625.21,739.43 601.83,739.43 C 578.46,739.43 547.09,736.12 518.47,729.19 C 489.85,722.26 456.08,707.75 429.46,689.16 C 402.83,670.57 380.05,656.52 353.64,631.62 C 327.23,606.72 323.27,599.75 313.08,588.93 C 302.90,578.11 281,557.43 281,557 Z',
    front_upper_teeth: 'M 416.27,523 C 417.33,523 415.49,538.00 417.21,540.00 C 418.92,542.00 464.57,551.50 467.17,550.15 C 469.78,548.79 470.04,531.94 471.16,532.01 C 471.92,532.06 472.03,547.63 474.50,550.15 C 476.97,552.66 526.73,555.94 529.20,553.97 C 531.66,552.00 532.52,533.01 533.15,533.01 C 533.79,533.01 533.99,552.68 536.35,554.97 C 538.71,557.26 592.03,559.15 595.62,557.26 C 599.21,555.37 598.86,531.01 600,531.01 C 601.14,531.01 601.02,555.23 604.22,557.26 C 607.41,559.29 662.18,558.29 663.97,555.97 C 665.76,553.65 665.55,531.08 666.73,531 C 667.91,530.92 667.92,551.82 670.73,553.97 C 673.54,556.12 722.12,553.61 725.39,551.15 C 728.66,548.67 726.82,531 728.12,531 C 729.42,531 729.77,547.38 732.14,550.15 C 734.51,552.91 783.12,544.14 784.30,541.86 C 785.47,539.59 783.14,523.01 784.30,523.01 C 785.45,523.01 787.75,538.86 789.34,540.81 C 790.93,542.75 832.19,547.05 839.58,547.86 C 846.96,548.68 896.24,552.59 893.00,550.64 C 889.76,548.68 861.45,537 839.58,529 C 817.70,521 789.34,502.44 756.23,489.80 C 723.12,477.17 697.47,466.84 687.98,464.87 C 678.49,462.90 638.00,455.90 625.25,461.89 C 612.50,467.88 604.93,475.14 601.91,475.14 C 599.17,475.14 582.30,462.77 570.16,457.87 C 558.01,452.97 521.48,465.39 508.92,468.08 C 496.35,470.76 427.98,500.13 416.27,504.08 C 404.56,508.03 364.79,524.90 348.62,533.01 C 337.84,538.41 321.28,545.36 298.93,553.86 C 338.47,553.77 360.42,553.10 364.79,551.86 C 371.35,550.00 407.48,540.86 410.28,538.65 C 413.07,536.43 415.22,523 416.27,523 Z',
    front_upper_mouth: 'M 309,544.07 C 300.82,547.75 281,556.75 281,557 C 281,557.25 293.01,558 293.86,558 C 294.70,558 334.49,558 350.68,558 C 366.87,558 405.65,550.76 421.82,548.63 C 437.98,546.50 473.42,544.47 497.32,544.47 C 521.22,544.47 556.40,540 600.24,540 C 644.08,540 684.66,542.50 705.77,544.07 C 726.88,545.65 761.63,546.50 779,548.63 C 796.37,550.77 840.67,557 850,557 C 856.89,557 895.74,558 898.51,558 C 901.28,558 918.72,557.16 918.72,557 C 918.72,556.37 898.73,546.79 893,544.07 C 887.27,541.36 763.02,489.72 742.08,480.68 C 721.15,471.63 704.68,466.11 682.27,458.90 C 659.87,451.68 640.94,450.79 627.43,453.73 C 613.93,456.67 604.30,467.11 601,467.11 C 597.70,467.11 588.71,458.90 573.48,454.33 C 558.26,449.77 530.47,455.96 520.09,458.90 C 509.71,461.84 496.57,466.86 467.52,478.58 C 438.48,490.30 317.18,540.40 309,544.07 Z'
  },

  n: {
    front_background: 'M 602.73,486.22 C 596.82,486.22 584.43,473.23 572.68,471.38 C 560.94,469.52 538.97,472.45 530.92,474.29 C 514.05,478.15 482.24,489.01 457.21,499.13 C 432.19,509.25 379.35,531.66 355.66,540.86 C 339.86,546.98 321.96,552.40 301.96,557.11 C 325.49,576.30 343.39,592.21 355.66,604.83 C 374.07,623.77 438.11,657.68 457.21,667.73 C 476.32,677.77 520.92,693.82 530.92,698.05 C 540.92,702.28 583.04,709.39 600,709.39 C 616.96,709.39 673.03,700.23 686.57,698.05 C 700.10,695.87 764.04,656.97 779,646.84 C 793.96,636.71 838.36,613.83 847.10,604.83 C 855.84,595.83 896.79,558.48 896.79,557.11 C 896.79,555.74 852.03,539.76 841.54,536.73 C 831.05,533.70 799.03,519.13 779,510.04 C 758.97,500.95 732.50,491.42 686.57,477.29 C 669.28,471.97 647.46,468.34 635.88,471.38 C 620.76,475.35 609.14,486.22 602.73,486.22 Z',
    front_tongue: 'M 407,607.79 C 407,602.41 430.19,582.39 452.15,571.87 C 474.11,561.34 562.52,530.53 570.03,526.59 C 577.54,522.64 585.70,516.38 602.36,516.38 C 615.03,516.38 631.64,523.01 638.26,526.59 C 644.87,530.17 729.56,561.57 749.80,571.87 C 770.04,582.16 800.46,602.77 800.46,612.87 C 800.46,622.97 771.38,654.57 749.80,670.27 C 728.22,685.97 705.52,694.21 692.34,700.23 C 684.56,703.78 620.28,713.48 603.73,713.48 C 584.87,713.48 533.46,705.89 519.71,700.23 C 500.23,692.21 476.27,677.94 452.15,663.44 C 428.03,648.94 407,613.17 407,607.79 Z',
    front_lower_teeth: 'M 315.76,581.38 C 315.76,580.78 365.40,589.30 375.77,592 C 386.15,594.70 418.99,595.61 420.66,598.00 C 422.33,600.39 418.39,618.49 419.66,618.67 C 420.94,618.84 424.79,600.66 426.50,599.83 C 428.21,599 458.77,603.67 461.24,606.49 C 463.71,609.32 463.71,631.19 464.42,631.19 C 465.13,631.19 465.14,609.66 468.28,607.06 C 471.42,604.45 500.16,607.49 501.84,610.32 C 503.53,613.14 503.43,639.25 504.60,639.29 C 505.64,639.32 505.48,612.24 508.09,610.48 C 510.70,608.72 543.09,609.34 545.62,611.96 C 548.16,614.57 547.31,642.83 548.39,642.83 C 549.46,642.83 548.83,614.51 551.11,612.48 C 553.40,610.45 596.25,610.32 597.79,612.96 C 599.34,615.59 599.03,648.83 600,648.83 C 600.95,648.83 601.52,615.19 603.67,612.96 C 605.83,610.72 646.61,610.83 649.00,613.48 C 651.39,616.13 651.50,642.83 652.52,642.83 C 653.54,642.83 652.20,614.62 654.93,612.48 C 657.66,610.34 688.58,607.49 691.75,609.72 C 694.93,611.96 696.60,637 697.93,637 C 699.26,637 695.93,611.96 697.88,609.34 C 699.83,606.73 729.58,604.11 732.55,606.75 C 735.52,609.39 735.29,632.46 736.29,632.43 C 737.21,632.40 736.50,609.08 738.29,606.58 C 740.09,604.08 774.69,603.93 775.13,604.36 C 775.57,604.78 778.47,622.66 779.69,622.66 C 780.56,622.66 780.03,604.25 781.00,603.36 C 781.97,602.46 814.87,600.40 824.57,598.00 C 831.04,596.39 850.87,590.85 884.05,581.38 C 859.30,604.54 843.65,619.15 837.11,625.19 C 827.30,634.25 803.80,652.11 786.40,662.82 C 769.00,673.52 740.98,691.13 719.16,702.74 C 697.34,714.36 630.77,724.89 603.73,724.89 C 576.70,724.89 529.20,717.30 508.09,711.70 C 486.98,706.09 433.99,674.65 419.66,662.26 C 405.34,649.87 390.58,641.85 374.77,628.35 C 358.98,614.86 315.76,581.98 315.76,581.38 Z',
    front_lower_mouth: 'M 281,557 C 281,556.57 286.18,556.46 287.34,557 C 288.50,557.54 304.92,555.90 309.84,557 C 314.76,558.10 342.18,561.06 361.52,575.73 C 380.88,590.39 432.53,616.34 461.24,622.94 C 489.94,629.54 547.19,642.73 601,645.39 C 654.81,648.05 712.33,637.74 745.95,622.94 C 779.56,608.14 820.05,592.96 848.66,575.73 C 877.26,558.49 892.75,558.24 895.53,558.01 C 898.30,557.77 910.09,556.90 915,557 C 916.09,557.02 918.63,556.87 918.72,557 C 918.81,557.13 907.34,568.01 900.79,575.73 C 894.24,583.45 879.29,599.83 862.88,614.39 C 846.47,628.95 811.43,657.39 787.27,673.75 C 763.11,690.11 719.84,716.40 691.48,723.47 C 663.12,730.54 626.11,734.39 602.73,734.39 C 579.36,734.39 545.29,732.86 512.38,723.47 C 479.47,714.08 441.78,692.72 414.67,673.75 C 387.56,654.78 373.41,642.17 356.63,626.89 C 339.85,611.60 319.01,591.16 309.84,583.59 C 300.67,576.03 281,557.43 281,557 Z',
    front_upper_teeth: 'M 417.28,525.96 C 418.11,525.96 419.88,540.56 420.75,540.56 C 421.63,540.56 464.57,540.50 467.17,539.15 C 469.78,537.79 470.04,520.94 471.16,521.01 C 471.92,521.06 472.03,536.63 474.50,539.15 C 476.97,541.66 526.73,544.94 529.20,542.97 C 531.66,541.00 532.52,522.01 533.15,522.01 C 533.79,522.01 533.99,541.68 536.35,543.97 C 538.71,546.26 592.03,548.15 595.62,546.26 C 599.21,544.37 598.86,520.01 600,520.01 C 601.14,520.01 601.03,544.23 604.22,546.26 C 607.41,548.29 662.18,547.29 663.97,544.97 C 665.76,542.65 665.55,520.08 666.73,520 C 667.91,519.92 667.92,540.82 670.73,542.97 C 673.54,545.12 722.12,542.61 725.39,540.15 C 728.66,537.67 726.78,520.01 728.08,520.01 C 729.38,520.01 729.77,536.38 732.14,539.15 C 734.51,541.91 779.12,543.14 780.30,540.86 C 781.47,538.59 780.29,528.67 781.45,528.67 C 782.61,528.67 783.75,537.86 785.34,539.81 C 786.93,541.75 827.60,548.05 837.77,551.00 C 847.93,553.95 896.79,552.12 896.79,551.00 C 896.79,549.87 844.17,528.67 822.33,519.55 C 800.49,510.43 774.02,499.82 752.69,490.51 C 731.36,481.20 695.46,469.60 670.73,465.61 C 646.01,461.63 623.07,469.67 614.40,475.27 C 605.73,480.87 605.52,485.77 602.50,485.77 C 599.76,485.77 585.51,469.10 572.16,465.61 C 558.80,462.13 535.54,462.93 522.97,465.61 C 510.41,468.30 433.84,500.57 413.87,507.57 C 393.91,514.58 365.34,526.98 346.05,534.63 C 330.19,540.92 314.71,547.17 299.63,553.38 C 327.58,552.58 343.05,551.78 346.05,551.00 C 350.54,549.82 415.77,541.72 417.28,541.00 C 418.79,540.28 416.44,525.96 417.28,525.96 Z',
    front_upper_mouth: 'M 329.37,533.21 C 315.81,539.79 281,556.75 281,557 C 281,557.25 290.57,558.03 294.17,558.03 C 297.77,558.03 315.22,558.03 329.37,558.03 C 343.51,558.03 404.29,553.73 425.17,548.92 C 446.05,544.11 494.16,537.70 517.79,537.70 C 541.43,537.70 570.93,537.70 603.73,537.70 C 636.53,537.70 669.86,537.70 695.34,537.70 C 720.83,537.70 763.58,544.83 785.27,548.92 C 806.95,553.01 841.98,555.60 864.53,556.64 C 887.09,557.67 903.32,558.03 905.02,558.03 C 906.72,558.03 918.72,557.17 918.72,557 C 918.72,556.37 864.37,531.81 858.64,529.10 C 852.91,526.39 782.70,494.90 754.16,483.34 C 725.61,471.77 716.05,466.36 695.34,459.27 C 674.64,452.18 656.34,453.76 643.67,456.27 C 631.00,458.78 604.81,475.22 602.73,475.22 C 600.66,475.22 579.90,458.18 566.11,456.27 C 552.31,454.36 535.08,453.19 511.60,459.27 C 488.12,465.35 473.48,474.52 441.04,486.99 C 408.59,499.46 342.92,526.63 329.37,533.21 Z'
  }
};

/** Default durations (seconds) for each phoneme, from the SMIL animation timing */
export const PHONEME_DURATIONS = {
  ow1: 0.0125,
  ow2: 0.0875,
  start: 0.0875,
  t: 0.0125,
  aa: 0.125,
  m: 0.0125,
  ax: 0.0625,
  n: 0.1625
};

/** The demo sequence from the original SMIL animation */
export const DEMO_SEQUENCE = [
  { phoneme: 'ow1', duration: 0.0125 },
  { phoneme: 'ow2', duration: 0.0875 },
  { phoneme: 'start', duration: 0.0875 },
  { phoneme: 't', duration: 0.0125 },
  { phoneme: 'aa', duration: 0.125 },
  { phoneme: 'start', duration: 0.1125 },
  { phoneme: 'm', duration: 0.0125 },
  { phoneme: 'ax', duration: 0.0625 },
  { phoneme: 'start', duration: 0.075 },
  { phoneme: 't', duration: 0.0125 },
  { phoneme: 'aa', duration: 0.1 },
  { phoneme: 'n', duration: 0.1625 },
  { phoneme: 'start', duration: 0.2125 }
];
