/**
 * workspace-panel.mjs — Workspace management component
 * Lists workspaces, switches active, creates/deletes workspaces, and browses files.
 */

import store from '../store.mjs';
import api from '../api.mjs';
import { $, $$, createElement, icon, escapeHtml, empty } from '../utils/dom.mjs';
import { relativeTime, formatBytes, truncate } from '../utils/format.mjs';
import { show as showModal, close as closeModal } from './modal.mjs';

// ── DOM references ──

let container = null;
let workspaceListEl = null;
let fileBrowserEl = null;
let fileBrowserTitle = null;
let fileListEl = null;
let fileEmptyEl = null;

// ──────────────────────────────────────
// Public init
// ──────────────────────────────────────

export function init() {
  container = $('#view-workspaces');
  if (!container) {
    console.warn('[workspace-panel] #view-workspaces not found');
    return;
  }

  buildDOM();
  subscribeToStore();
  renderWorkspaces();
  loadFiles();
}

// ──────────────────────────────────────
// DOM construction
// ──────────────────────────────────────

function buildDOM() {
  empty(container);
  container.classList.add('workspace-panel');

  // Header
  const header = createElement('div', {
    style: {
      display: 'flex', alignItems: 'center', justifyContent: 'space-between',
      flexWrap: 'wrap', gap: 'var(--space-sm)'
    }
  });

  const titleRow = createElement('div', {
    style: { display: 'flex', alignItems: 'center', gap: 'var(--space-sm)' }
  }, [
    icon('folder', { size: 24 }),
    createElement('h2', {
      style: { margin: '0', fontSize: 'var(--font-size-xl)', color: 'var(--text-primary)' }
    }, ['Workspaces'])
  ]);

  const newBtn = createElement('button', {
    className: 'glass-button glass-button-accent',
    style: { display: 'flex', alignItems: 'center', gap: 'var(--space-xs)' }
  }, [
    icon('plus', { size: 16 }),
    'New Workspace'
  ]);
  newBtn.addEventListener('click', showCreateModal);

  header.appendChild(titleRow);
  header.appendChild(newBtn);
  container.appendChild(header);

  // Workspace list
  workspaceListEl = createElement('div', {
    style: { display: 'flex', flexDirection: 'column', gap: 'var(--space-sm)' }
  });
  container.appendChild(workspaceListEl);

  // Divider
  container.appendChild(createElement('div', { className: 'divider' }));

  // File browser
  fileBrowserEl = createElement('div', {
    style: { display: 'flex', flexDirection: 'column', gap: 'var(--space-sm)', flex: '1', minHeight: '0' }
  });

  const fileHeader = createElement('div', {
    style: {
      display: 'flex', alignItems: 'center', justifyContent: 'space-between',
      gap: 'var(--space-sm)'
    }
  });

  fileBrowserTitle = createElement('div', {
    style: {
      display: 'flex', alignItems: 'center', gap: 'var(--space-sm)',
      color: 'var(--text-secondary)', fontSize: 'var(--font-size-sm)', fontWeight: '500'
    }
  }, [
    icon('file', { size: 16 }),
    'Files'
  ]);

  const refreshBtn = createElement('button', {
    className: 'glass-button-icon glass-button-sm',
    'aria-label': 'Refresh files',
    style: { display: 'flex', alignItems: 'center', justifyContent: 'center' }
  }, [icon('refresh', { size: 14 })]);
  refreshBtn.addEventListener('click', loadFiles);

  fileHeader.appendChild(fileBrowserTitle);
  fileHeader.appendChild(refreshBtn);
  fileBrowserEl.appendChild(fileHeader);

  // File list container
  fileListEl = createElement('div', {
    style: { display: 'flex', flexDirection: 'column', gap: 'var(--space-xs)', overflowY: 'auto' }
  });
  fileBrowserEl.appendChild(fileListEl);

  // File empty state
  fileEmptyEl = createElement('div', {
    className: 'glass-subtle',
    style: {
      display: 'none', flexDirection: 'column', alignItems: 'center', justifyContent: 'center',
      padding: 'var(--space-lg)', gap: 'var(--space-sm)', textAlign: 'center',
      borderRadius: 'var(--radius-md)', minHeight: '120px'
    }
  }, [
    icon('file', { size: 32 }),
    createElement('p', {
      style: { color: 'var(--text-secondary)', margin: '0', fontSize: 'var(--font-size-sm)' }
    }, ['No files yet']),
    createElement('p', {
      style: { color: 'var(--text-tertiary)', margin: '0', fontSize: 'var(--font-size-xs)' }
    }, ['Files created by skills will appear here'])
  ]);
  fileEmptyEl.querySelector('svg').style.color = 'var(--text-tertiary)';
  fileBrowserEl.appendChild(fileEmptyEl);

  container.appendChild(fileBrowserEl);
}

// ──────────────────────────────────────
// Store subscriptions
// ──────────────────────────────────────

function subscribeToStore() {
  store.subscribe(
    (s) => s.workspaces.list,
    () => renderWorkspaces()
  );

  store.subscribe(
    (s) => s.workspaces.active,
    () => {
      renderWorkspaces();
      loadFiles();
    }
  );
}

// ──────────────────────────────────────
// Rendering — Workspace list
// ──────────────────────────────────────

function renderWorkspaces() {
  if (!workspaceListEl) return;

  const { list, active } = store.getState().workspaces;
  empty(workspaceListEl);

  if (!list || list.length === 0) {
    workspaceListEl.appendChild(
      createElement('div', {
        className: 'glass-subtle',
        style: {
          padding: 'var(--space-md)', textAlign: 'center',
          borderRadius: 'var(--radius-md)', color: 'var(--text-secondary)'
        }
      }, ['No workspaces found'])
    );
    return;
  }

  for (const ws of list) {
    const card = buildWorkspaceCard(ws, active);
    workspaceListEl.appendChild(card);
  }
}

function buildWorkspaceCard(ws, activeId) {
  const isActive = ws.id === activeId || ws.name === activeId;
  const isDefault = (ws.name || '').toLowerCase() === 'default' || ws.id === 'default';

  const card = createElement('div', {
    className: `glass-card animate-fade-in`,
    style: {
      padding: 'var(--space-md)',
      borderRadius: 'var(--radius-md)',
      cursor: isActive ? 'default' : 'pointer',
      transition: 'var(--transition-fast)',
      borderColor: isActive ? 'var(--accent-primary)' : '',
      borderWidth: isActive ? '1px' : '',
      borderStyle: isActive ? 'solid' : ''
    },
    dataset: { id: String(ws.id || ''), active: isActive ? 'true' : '' }
  });

  // Header row: name + active badge
  const headerRow = createElement('div', {
    style: {
      display: 'flex', alignItems: 'center', gap: 'var(--space-sm)',
      marginBottom: 'var(--space-xs)'
    }
  });

  const nameEl = createElement('span', {
    style: {
      fontWeight: '500', color: 'var(--text-primary)',
      fontSize: 'var(--font-size-base)'
    }
  }, [escapeHtml(ws.name || 'Unnamed')]);
  headerRow.appendChild(nameEl);

  if (isActive) {
    headerRow.appendChild(
      createElement('span', { className: 'badge badge-success' }, ['Active'])
    );
  }

  card.appendChild(headerRow);

  // Description
  if (ws.description) {
    const descEl = createElement('div', {
      style: {
        color: 'var(--text-secondary)', fontSize: 'var(--font-size-sm)',
        lineHeight: '1.4', marginBottom: 'var(--space-sm)'
      }
    }, [truncate(ws.description, 120)]);
    card.appendChild(descEl);
  }

  // Footer: file count + delete button
  const footer = createElement('div', {
    style: {
      display: 'flex', alignItems: 'center', justifyContent: 'space-between',
      marginTop: 'var(--space-xs)'
    }
  });

  const fileCount = ws.fileCount != null ? ws.fileCount : (ws.files || 0);
  footer.appendChild(
    createElement('span', {
      style: { color: 'var(--text-tertiary)', fontSize: 'var(--font-size-xs)' }
    }, [`${fileCount} file${fileCount !== 1 ? 's' : ''}`])
  );

  if (!isDefault) {
    const deleteBtn = createElement('button', {
      className: 'glass-button-icon glass-button-sm',
      'aria-label': `Delete workspace ${ws.name}`,
      style: { color: 'var(--error)', display: 'flex', alignItems: 'center', justifyContent: 'center' }
    }, [icon('trash', { size: 14 })]);

    deleteBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      confirmDelete(ws);
    });

    footer.appendChild(deleteBtn);
  }

  card.appendChild(footer);

  // Click to activate (if not already active)
  if (!isActive) {
    card.addEventListener('mouseenter', () => {
      card.style.background = 'var(--glass-bg-hover)';
    });
    card.addEventListener('mouseleave', () => {
      card.style.background = '';
    });
    card.addEventListener('click', () => activateWorkspace(ws.id));
  }

  return card;
}

// ──────────────────────────────────────
// Workspace actions
// ──────────────────────────────────────

async function activateWorkspace(id) {
  try {
    await api.activateWorkspace(id);
    store.dispatch('SET_ACTIVE_WORKSPACE', id);
    await refreshWorkspaceList();
  } catch (err) {
    console.error('[workspace-panel] Activate error:', err);
  }
}

function showCreateModal() {
  const form = createElement('div', {
    style: { display: 'flex', flexDirection: 'column', gap: 'var(--space-md)' }
  });

  const nameLabel = createElement('label', {
    style: { color: 'var(--text-secondary)', fontSize: 'var(--font-size-sm)', fontWeight: '500' }
  }, ['Name']);
  const nameInput = createElement('input', {
    className: 'glass-input',
    type: 'text',
    placeholder: 'Workspace name',
    'aria-label': 'Workspace name',
    style: { width: '100%' }
  });

  const descLabel = createElement('label', {
    style: { color: 'var(--text-secondary)', fontSize: 'var(--font-size-sm)', fontWeight: '500' }
  }, ['Description (optional)']);
  const descInput = createElement('textarea', {
    className: 'glass-textarea',
    placeholder: 'What is this workspace for?',
    'aria-label': 'Workspace description',
    rows: '3',
    style: { width: '100%', resize: 'vertical' }
  });

  form.appendChild(nameLabel);
  form.appendChild(nameInput);
  form.appendChild(descLabel);
  form.appendChild(descInput);

  showModal({
    title: 'New Workspace',
    content: form,
    closable: true,
    actions: [
      {
        label: 'Cancel',
        className: 'glass-button',
        onClick: () => closeModal()
      },
      {
        label: 'Create',
        className: 'glass-button glass-button-accent',
        onClick: async () => {
          const name = nameInput.value.trim();
          if (!name) {
            nameInput.style.borderColor = 'var(--error)';
            nameInput.focus();
            return;
          }
          try {
            await api.createWorkspace({ name, description: descInput.value.trim() });
            closeModal();
            await refreshWorkspaceList();
          } catch (err) {
            console.error('[workspace-panel] Create error:', err);
          }
        }
      }
    ]
  });

  // Focus the name input after modal animation
  requestAnimationFrame(() => nameInput.focus());
}

function confirmDelete(ws) {
  const message = createElement('p', {
    style: { color: 'var(--text-secondary)', margin: '0' }
  }, [`Are you sure you want to delete the workspace "${escapeHtml(ws.name)}"? This action cannot be undone.`]);

  showModal({
    title: 'Delete Workspace',
    content: message,
    closable: true,
    actions: [
      {
        label: 'Cancel',
        className: 'glass-button',
        onClick: () => closeModal()
      },
      {
        label: 'Delete',
        className: 'glass-button',
        onClick: async () => {
          try {
            await api.deleteWorkspace(ws.id);
            closeModal();
            await refreshWorkspaceList();
          } catch (err) {
            console.error('[workspace-panel] Delete error:', err);
          }
        }
      }
    ]
  });
}

async function refreshWorkspaceList() {
  try {
    const ws = await api.getWorkspaces();
    store.dispatch('SET_WORKSPACES', ws?.workspaces || ws || []);
    if (ws?.active) store.dispatch('SET_ACTIVE_WORKSPACE', ws.active);
  } catch (err) {
    console.error('[workspace-panel] Refresh error:', err);
  }
}

// ──────────────────────────────────────
// File browser
// ──────────────────────────────────────

async function loadFiles() {
  if (!fileListEl) return;

  // Update file browser title with active workspace name
  updateFileBrowserTitle();

  empty(fileListEl);
  fileEmptyEl.style.display = 'none';

  // Show shimmer while loading
  const shimmer = createElement('div', {
    className: 'shimmer',
    style: { height: '40px', borderRadius: 'var(--radius-sm)' }
  });
  fileListEl.appendChild(shimmer);

  try {
    const files = await api.getWorkspaceFiles();
    empty(fileListEl);

    if (!files || files.length === 0) {
      fileEmptyEl.style.display = 'flex';
      return;
    }

    for (const file of files) {
      const row = buildFileRow(file);
      fileListEl.appendChild(row);
    }
  } catch (err) {
    console.warn('[workspace-panel] Files load error:', err.message);
    empty(fileListEl);
    fileEmptyEl.style.display = 'flex';
  }
}

function updateFileBrowserTitle() {
  if (!fileBrowserTitle) return;

  const { list, active } = store.getState().workspaces;
  const activeWs = (list || []).find(w => w.id === active || w.name === active);
  const name = activeWs ? activeWs.name : 'Workspace';

  empty(fileBrowserTitle);
  fileBrowserTitle.appendChild(icon('file', { size: 16 }));
  fileBrowserTitle.appendChild(document.createTextNode(`Files in ${name}`));
}

function buildFileRow(file) {
  const isOpenable = /\.(html?|txt|md|css|js|mjs|json|csv)$/i.test(file.name || '');

  const row = createElement('div', {
    className: 'glass-subtle',
    style: {
      display: 'flex', alignItems: 'center', gap: 'var(--space-sm)',
      padding: 'var(--space-sm) var(--space-md)',
      borderRadius: 'var(--radius-sm)',
      cursor: isOpenable ? 'pointer' : 'default',
      transition: 'var(--transition-fast)'
    }
  });

  // File icon
  const iconEl = createElement('span', {
    style: { display: 'flex', color: 'var(--text-tertiary)', flexShrink: '0' }
  }, [icon('file', { size: 16 })]);

  // File name
  const nameEl = createElement('span', {
    style: {
      flex: '1', color: 'var(--text-primary)', fontSize: 'var(--font-size-sm)',
      overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap'
    }
  }, [escapeHtml(file.name || 'unnamed')]);

  // File size
  const sizeEl = createElement('span', {
    style: {
      color: 'var(--text-tertiary)', fontSize: 'var(--font-size-xs)',
      whiteSpace: 'nowrap', flexShrink: '0'
    }
  }, [file.size != null ? formatBytes(file.size) : '']);

  // Modified time
  const modEl = createElement('span', {
    style: {
      color: 'var(--text-tertiary)', fontSize: 'var(--font-size-xs)',
      whiteSpace: 'nowrap', flexShrink: '0'
    }
  }, [file.modified ? relativeTime(file.modified) : '']);

  row.appendChild(iconEl);
  row.appendChild(nameEl);
  row.appendChild(sizeEl);
  row.appendChild(modEl);

  if (isOpenable) {
    row.addEventListener('mouseenter', () => {
      row.style.background = 'var(--glass-bg-hover)';
    });
    row.addEventListener('mouseleave', () => {
      row.style.background = '';
    });
    row.addEventListener('click', () => {
      const path = file.publicPath || `/workspace-files/${encodeURIComponent(file.name)}`;
      window.open(path, '_blank');
    });
  }

  return row;
}
