// Knowledge Graph Explorer UI component
export function init({ apiFetch, container }) {
  let entities = [];
  let selectedEntity = null;
  let relations = [];
  let stats = { entityCount: 0, relationCount: 0 };

  async function loadStats() {
    try {
      const res = await apiFetch('/api/kg/stats');
      stats = await res.json();
    } catch { stats = { entityCount: 0, relationCount: 0 }; }
  }

  async function searchEntities(query) {
    try {
      const params = query ? `?q=${encodeURIComponent(query)}&limit=50` : '?limit=50';
      const res = await apiFetch(`/api/kg/entities${params}`);
      entities = await res.json();
    } catch { entities = []; }
    selectedEntity = null;
    relations = [];
    render();
  }

  async function selectEntity(entityId) {
    try {
      const res = await apiFetch(`/api/kg/entities/${entityId}/relations`);
      relations = await res.json();
    } catch { relations = []; }
    selectedEntity = entities.find(e => e.id === entityId) || null;
    render();
  }

  const TYPE_COLORS = {
    person: '#58a6ff', technology: '#3fb950', concept: '#bc8cff',
    organization: '#d18616', place: '#d29922', event: '#f85149',
    document: '#8b949e', insight: '#bc8cff', other: '#30363d'
  };

  function render() {
    container.innerHTML = `
      <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:16px;">
        <h2 style="color:var(--accent);font-size:16px;margin:0;">Knowledge Graph</h2>
        <span style="color:var(--text-dim);font-size:11px;">${stats.entityCount} entities, ${stats.relationCount} relations</span>
      </div>
      <div style="margin-bottom:16px;">
        <input type="text" id="kg-search" placeholder="Search entities..." value=""
               style="width:100%;padding:8px 12px;background:var(--bg3);border:1px solid var(--border);border-radius:6px;color:var(--text);font-family:var(--mono);font-size:12px;outline:none;box-sizing:border-box;">
      </div>
      <div style="display:flex;gap:16px;min-height:200px;">
        <div style="flex:1;overflow-y:auto;max-height:400px;">
          <div style="font-size:11px;color:var(--text-dim);text-transform:uppercase;letter-spacing:1px;margin-bottom:8px;">Entities (${entities.length})</div>
          ${entities.length ? entities.map(renderEntity).join('') : '<div style="color:var(--text-dim);font-size:12px;">No entities found. Ingest content to populate the knowledge graph.</div>'}
        </div>
        <div style="flex:1;overflow-y:auto;max-height:400px;">
          <div style="font-size:11px;color:var(--text-dim);text-transform:uppercase;letter-spacing:1px;margin-bottom:8px;">
            ${selectedEntity ? `Relations for: ${selectedEntity.name}` : 'Select an entity'}
          </div>
          ${selectedEntity ? renderRelations() : '<div style="color:var(--text-dim);font-size:12px;">Click an entity to view its relations</div>'}
        </div>
      </div>
      ${selectedEntity && relations.length ? `
      <div style="margin-top:24px;">
        <div style="font-size:11px;color:var(--text-dim);text-transform:uppercase;letter-spacing:1px;margin-bottom:8px;">Graph View</div>
        <div id="kg-graph" style="background:var(--bg2);border:1px solid var(--border);border-radius:6px;height:300px;overflow:hidden;"></div>
      </div>` : ''}
    `;

    // Search input with debounce
    const searchInput = container.querySelector('#kg-search');
    let searchTimer = null;
    searchInput?.addEventListener('input', () => {
      if (searchTimer) clearTimeout(searchTimer);
      searchTimer = setTimeout(() => searchEntities(searchInput.value), 300);
    });

    // Entity click handlers
    container.querySelectorAll('[data-entity-id]').forEach(el => {
      el.addEventListener('click', () => selectEntity(parseInt(el.dataset.entityId)));
    });

    // Render graph if applicable
    if (selectedEntity && relations.length) {
      setTimeout(() => renderGraph(), 50);
    }
  }

  function renderEntity(e) {
    const isSelected = selectedEntity?.id === e.id;
    const color = TYPE_COLORS[e.type] || TYPE_COLORS.other;
    return `<div data-entity-id="${e.id}" style="padding:8px 10px;border:1px solid ${isSelected ? 'var(--accent)' : 'var(--border)'};border-radius:4px;margin-bottom:4px;cursor:pointer;background:${isSelected ? 'var(--bg3)' : 'transparent'};">
      <span style="color:var(--text);">${e.name}</span>
      <span style="font-size:9px;padding:1px 5px;border-radius:8px;border:1px solid ${color};color:${color};margin-left:6px;">${e.type}</span>
    </div>`;
  }

  function renderRelations() {
    if (!relations.length) return '<div style="color:var(--text-dim);font-size:12px;">No relations found</div>';
    return relations.map(r => {
      const isOutgoing = r.source_id === selectedEntity.id;
      const otherName = isOutgoing ? r.target_name : r.source_name;
      const arrow = isOutgoing ? '&rarr;' : '&larr;';
      return `<div style="padding:6px 10px;border-left:3px solid var(--accent2);margin-bottom:4px;background:var(--bg2);border-radius:0 4px 4px 0;">
        <span style="color:var(--text-dim);font-size:10px;">${arrow} ${r.relation}</span>
        <span style="color:var(--text);margin-left:6px;">${otherName || '?'}</span>
        <span style="color:var(--text-dim);font-size:10px;margin-left:6px;">w:${(r.weight || 1).toFixed(1)}</span>
      </div>`;
    }).join('');
  }

  function renderGraph() {
    const graphEl = container.querySelector('#kg-graph');
    if (!graphEl || !selectedEntity) return;

    const w = graphEl.clientWidth || 500;
    const h = 300;
    const cx = w / 2;
    const cy = h / 2;

    // Build nodes
    const nodes = new Map();
    nodes.set(selectedEntity.id, { ...selectedEntity, x: cx, y: cy });
    for (const r of relations) {
      if (!nodes.has(r.source_id)) nodes.set(r.source_id, { id: r.source_id, name: r.source_name || '?', type: r.source_type || 'other', x: 0, y: 0 });
      if (!nodes.has(r.target_id)) nodes.set(r.target_id, { id: r.target_id, name: r.target_name || '?', type: r.target_type || 'other', x: 0, y: 0 });
    }

    // Circular layout for non-center nodes
    const others = [...nodes.values()].filter(n => n.id !== selectedEntity.id);
    const radius = Math.min(w, h) * 0.35;
    others.forEach((n, i) => {
      const angle = (2 * Math.PI * i) / Math.max(others.length, 1);
      n.x = cx + radius * Math.cos(angle);
      n.y = cy + radius * Math.sin(angle);
    });

    // SVG
    let svg = `<svg width="${w}" height="${h}" viewBox="0 0 ${w} ${h}" style="width:100%;height:100%;">`;

    // Edges
    for (const r of relations) {
      const src = nodes.get(r.source_id);
      const tgt = nodes.get(r.target_id);
      if (src && tgt) {
        svg += `<line x1="${src.x}" y1="${src.y}" x2="${tgt.x}" y2="${tgt.y}" stroke="#30363d" stroke-width="1" opacity="0.6"/>`;
        const mx = (src.x + tgt.x) / 2;
        const my = (src.y + tgt.y) / 2;
        svg += `<text x="${mx}" y="${my - 4}" text-anchor="middle" fill="#8b949e" font-size="9" font-family="'SF Mono',monospace">${r.relation}</text>`;
      }
    }

    // Nodes
    for (const n of nodes.values()) {
      const isCenter = n.id === selectedEntity.id;
      const r = isCenter ? 8 : 6;
      const color = TYPE_COLORS[n.type] || TYPE_COLORS.other;
      svg += `<circle cx="${n.x}" cy="${n.y}" r="${r}" fill="${color}" opacity="0.8"/>`;
      svg += `<text x="${n.x}" y="${n.y + r + 12}" text-anchor="middle" fill="#e6edf3" font-size="10" font-family="'SF Mono',monospace">${(n.name || '').slice(0, 20)}</text>`;
    }

    svg += '</svg>';
    graphEl.innerHTML = svg;
  }

  // Initial load
  loadStats().then(() => searchEntities(''));
  return { refresh: () => loadStats().then(() => searchEntities('')) };
}
