// NOTE: This skill creates files. 'research' must be added to FILE_CREATING_SKILLS in skills.mjs during Phase 3 integration.
import { fetchUrl, htmlToMarkdown, extractMetadata } from '../fetch-utils.mjs';
import { ingestText } from '../ingestion.mjs';

export const name = 'research';
export const description = 'Research a topic by fetching and synthesizing information from multiple sources';

export async function execute(ctx) {
  const topic = ctx.args?.topic || ctx.args?.task || '';
  if (!topic) return { success: false, reply: 'No topic provided for research.' };
  const workspaceId = ctx.getActiveWorkspaceId();

  // Step 1: Ask LLM for search URLs
  const urlPrompt = `Suggest 3-5 web URLs to research: "${topic}". Return JSON: { urls: ["https://..."] }. Only include real, likely-accessible URLs.`;
  let urls = [];
  try {
    const urlResponse = await ctx.askLLM(urlPrompt, { model: ctx.pickModel('ingest') });
    // Try to parse JSON from response
    const match = urlResponse.match(/\{[\s\S]*\}/);
    if (match) { const parsed = JSON.parse(match[0]); urls = parsed.urls || []; }
  } catch {}

  // Step 2: Fetch and convert each URL
  const sources = [];
  for (const url of urls.slice(0, 5)) {
    try {
      const fetched = await fetchUrl(url, { timeout: 10000 });
      const md = htmlToMarkdown(fetched.html);
      const meta = extractMetadata(fetched.html);
      sources.push({ url, markdown: md.slice(0, 3000), title: meta.title || url });
      // Ingest each source for KG
      try { await ingestText(md.slice(0, 5000), { workspaceId, extractEntities: true }); } catch {}
    } catch {}
  }

  // Step 3: Synthesize
  const synthesisPrompt = `Write a comprehensive research summary about: "${topic}"
Based on these sources:
${sources.map((s, i) => `\n### Source ${i + 1}: ${s.title}\n${s.markdown.slice(0, 1500)}`).join('\n')}

Write a well-structured markdown document with sections, key findings, and conclusions.${sources.length === 0 ? '\nNo sources could be fetched. Write based on your knowledge.' : ''}`;

  const summary = await ctx.askLLM(synthesisPrompt, { model: ctx.pickModel('build') });

  // Step 4: Write to workspace
  const slug = topic.toLowerCase().replace(/[^a-z0-9]+/g, '-').slice(0, 40);
  const written = await ctx.writeOutputFile(`research-${slug}.md`, summary);
  ctx.addEvent('skill', 'agent', `Research complete: ${written.name}`, { skill: name, path: written.publicPath, sources: sources.length });
  return { success: true, reply: `Research complete: ${written.name} (${sources.length} sources fetched)` };
}
